/*
 * TagTreeContextMenu.cpp
 *
 *  Created on: 03.05.2010
 *      Author: stefan.detter
 */

#include "TagTreeContextMenu.h"

#include <QApplication>
#include <QClipboard>
#include <QMenu>
#include <QAction>
#include <QPoint>
#include <QTreeView>
#include <QMessageBox>

#include <QrfeResourceGlobal>

#include "TagTreeModel.h"
#include "TagTreeProxy.h"

#include "../../application/Reader.h"

#include "../data/TreeItem.h"
#include "../data/RootItem.h"
#include "../data/ReaderEntry.h"
#include "../data/AntennaEntry.h"
#include "../data/TagEntry.h"

#include "../../permission/Permissions.h"

#include <def.h>

TagTreeContextMenu::TagTreeContextMenu(QTreeView* treeView, TagTreeModel* tagTreeModel, TagTreeProxy* tagTreeProxy, QWidget* window, QObject* parent)
	: QObject(parent)
	, m_treeView(treeView)
	, m_tagTreeModel(tagTreeModel)
	, m_tagTreeProxy(tagTreeProxy)
	, m_window(window)
{
	// create context menus and actions
	m_treeView->setContextMenuPolicy(Qt::CustomContextMenu);
	m_context = new QMenu("Tag Context Menu", m_treeView);

	m_context_Settings = new QAction("Settings", this);
	m_context_Settings->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("preferences")));
	connect(m_context_Settings, SIGNAL(triggered (bool)), this, SLOT(settingsRequested()));

	m_context_AppSettings = new QAction("Application Settings", this);
	m_context_AppSettings->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("settings")));
	connect(m_context_AppSettings, SIGNAL(triggered (bool)), this, SLOT(settingsRequested()));

	m_context_AddReader = new QAction("Add reader", this);
	m_context_AddReader->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("add")));
	connect(m_context_AddReader, SIGNAL(triggered (bool)), this, SIGNAL(addReaderDialogRequested()));

	m_context_RemoveReader = new QAction("Remove reader", this);
	m_context_RemoveReader->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("minus")));
	connect(m_context_RemoveReader, SIGNAL(triggered (bool)), this, SLOT(removeReaderRequested()));

	m_context_ClearTags = new QAction("Clear Tags", this);
	m_context_ClearTags->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("clear")));
	connect(m_context_ClearTags, SIGNAL(triggered (bool)), m_tagTreeModel, SLOT(clearTags()));

	m_context_Pipe = new QAction("Pipe Editor", this);
	m_context_Pipe->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("pipe")));
	connect(m_context_Pipe, SIGNAL(triggered (bool)), this, SLOT(settingsRequested()));

	m_context_Engineering = new QAction("Engineering Mode", this);
	m_context_Engineering->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("engineering")));
	connect(m_context_Engineering, SIGNAL(triggered (bool)), this, SLOT(settingsRequested()));

	m_context_CopyTagId = new QAction("Copy TagId to Clipboard", this);
	m_context_CopyTagId->setIcon(QIcon(QrfeGlobal::getButtonIconResourcePath("copy")));
	connect(m_context_CopyTagId, SIGNAL(triggered (bool)), this, SLOT(copyToClipboardRequested()));
}

TagTreeContextMenu::~TagTreeContextMenu()
{
}

void TagTreeContextMenu::showPopup(const QPoint & iPoint)
{
	// save the item, on which was clicked in a member
	m_contextIndex = m_treeView->indexAt(iPoint);

	TreeItem* t;
	// if there was no item saved, return
	if ( !m_contextIndex.isValid() )
		t = (TreeItem*)m_tagTreeModel->rootItem();
	else
		t = static_cast<TreeItem*>(m_tagTreeProxy->mapToSource(m_contextIndex).internalPointer());

	m_context->clear();
	switch(t->itemType())
	{
	case TreeItem::ROOT:
		m_context->addAction(m_context_AddReader);
		m_context->addAction(m_context_Settings);
		m_context->addAction(m_context_ClearTags);
		break;
	case TreeItem::READER:
	{
		ReaderEntry* r = dynamic_cast<ReaderEntry*>(t);
		if(r == 0)
			return;
		ulong caps = r->reader()->readerCaps();

		if( (caps & QrfeGlobal::READER_SETTINGS) || (caps & QrfeGlobal::UPDATE_DIALOG) )
			m_context->addAction(m_context_Settings);
		if( (caps & QrfeGlobal::APPLICATION_SETTINGS) )
			m_context->addAction(m_context_AppSettings);

		m_context->addSeparator();

		if( (caps & QrfeGlobal::ENGINEERING) && (Permissions::d->readerAnyEngineering()) )
		{
			m_context->addAction(m_context_Engineering);
		}

		m_context->addSeparator();

		m_context->addAction(m_context_RemoveReader);
		break;
	}
	case TreeItem::TAG:
	{
		TagEntry* tag = dynamic_cast<TagEntry*>(t);
		if(tag == 0) return;
		ReaderEntry* r = tag->reader();

		m_context->addAction(m_context_Settings);

		m_context->addSeparator();

		if(r->reader()->readerType() == QrfeGlobal::TYPE_PASSIVE && Permissions::d->tagEngineering()){
			m_context->addAction(m_context_Engineering);
			m_context->addSeparator();
		}

		m_context->addAction(m_context_CopyTagId);

		break;
	}
	default:
		return;
	}
	m_context->exec(QPoint(m_treeView->mapToGlobal(iPoint).x()+3, m_treeView->mapToGlobal(iPoint).y()+25));
}

void TagTreeContextMenu::settingsRequested()
{
	TreeItem* t;
	// if there was no item saved, return
	if ( !m_contextIndex.isValid() )
		t = (TreeItem*)m_tagTreeModel->rootItem();
	else
		t = static_cast<TreeItem*>(m_tagTreeProxy->mapToSource(m_contextIndex).internalPointer());

	QList<QVariant> params;
	switch(t->itemType())
	{
		case TreeItem::ROOT:
		{
			emit appendGlobalJob(JOB_ShowApplicationSettings, params);
			break;
		}
		case TreeItem::READER:
		{
			ReaderEntry* r = (ReaderEntry*) t;
			params.append(r->readerId());
			if(sender() == m_context_Settings)
				emit appendGlobalJob(JOB_ShowSettingsOfReader, params);
			else if(sender() == m_context_AppSettings)
				emit appendGlobalJob(JOB_ShowAppSettingsOfReader, params);
			else if(sender() == m_context_Pipe)
				emit appendGlobalJob(JOB_ShowPipeForReader, params);
			else if(sender() == m_context_Engineering)
				emit appendGlobalJob(JOB_ShowEngineeringForReader, params);
			break;
		}

		case TreeItem::TAG:
		{
			TagEntry* tag = (TagEntry*) t;
			ReaderEntry* r = (ReaderEntry*) tag->reader();

			params.append(r->readerId());
			params.append(tag->tagId());
			params.append(tag->antennaId());

			if(sender() == m_context_Settings)
				emit appendGlobalJob(JOB_ShowSettingsOfTag, params);
			else if(sender() == m_context_Engineering)
				emit appendGlobalJob(JOB_ShowEngineeringForTag, params);
			break;
		}
		default:
			break;
	}

}

void TagTreeContextMenu::removeReaderRequested()
{
	TreeItem* t;
	// if there was no item saved, return
	if ( !m_contextIndex.isValid() )
		t = (TreeItem*)m_tagTreeModel->rootItem();
	else
		t = static_cast<TreeItem*>(m_tagTreeProxy->mapToSource(m_contextIndex).internalPointer());

	if(t->itemType() != TreeItem::READER)
		return;

	QMessageBox::StandardButton b;
	b = QMessageBox::question(
					m_window,
					"Remove Reader",
					"Do you want to remove the selected reader?",
					QMessageBox::Yes | QMessageBox::No,
					QMessageBox::No);

	if(b != QMessageBox::Yes)
		return;

	RootItem* 		root = (RootItem*) t->parent();
	ReaderEntry*	reader = (ReaderEntry*) t;

	root->removeReader(reader->readerId());
}

void TagTreeContextMenu::copyToClipboardRequested()
{
	TreeItem* t;
	// if there was no item saved, return
	if ( !m_contextIndex.isValid() )
		t = (TreeItem*)m_tagTreeModel->rootItem();
	else
		t = static_cast<TreeItem*>(m_tagTreeProxy->mapToSource(m_contextIndex).internalPointer());

	if(t->itemType() != TreeItem::TAG)
		return;

	TagEntry* tag = (TagEntry*)t;

    QApplication::clipboard()->setText(tag->visualTagId());
}
